/*
* \file: errmemd_backend.h
*
* Definition of the backend interface.
* 
* This file defines the backend interface structure as well as
* the prototypes of the interface functions possibly provided
* by the different backends.
*
* \component: errmemd
*
* \author: Kai Tomerius (ktomerius@de.adit-jv.com)
*          Markus Kretschmann (mkretschmann@de.adit-jv.com)
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
* <history item>
*/

#ifndef _ERRMEM_BACKEND_H
#define _ERRMEM_BACKEND_H

#include <stdio.h>
#include <stdint.h>
#include <linux/errmem.h>

//#include "errmem_socket_interface.h"

#define NOR_FLASH_CYCLE_TIME 60

/* backend type */
#define UNDEFINED       0x0

/* device types for persistent storages defined here
 * device types for non persistent storages defined in
 * errmem_lib/include/errmem_lib.h
 * Note: Guarantee that these values do not match the
 * values defined for non persistent devices
 */
#define FILE_FS         0x1    /* output to files on a fs        - opt: -f    */
#define BLOCK           0x2    /* output to a mounted block dev  - opt: -s -o */
#define BLOCK_RAW       0x4    /* output to a raw block device   - opt: -s    */
#define BLOCK_FLASH     0x8    /* init as long flash type unknown- opt: -r -o */
#define BLOCK_FLASH_RAW 0x10   /* init as long flash type unknown- opt: -r    */
#define BLOCK_NOR       0x20   /* output to a mounted NOR Flash  - opt: -r -o */
#define BLOCK_NOR_RAW   0x40   /* output to a raw NOR Flash      - opt: -r    */
#define BLOCK_NAND      0x100  /* output to a mounted NAND Flash - opt: -r -o */
#define BLOCK_NAND_RAW  0x200  /* output to a raw NAND Flas      - opt: -r    */
#define BUFFER          0x400  /* output is a data buffer        - internal   */


/* backend access options */
#define OPT_ACC_NONE   0x0
#define OPT_ACC_CREATE 0x1
#define OPT_ACC_BACKUP 0x2
#define OPT_ACC_END    0x4

/* device status */
#define STAT_UNKNOWN   0x0
#define STAT_OK        0x1
#define STAT_ERASED    0x2
#define STAT_CREATED   0x3
#define STAT_ABORT     0x4
#define STAT_CREATE    0x5
#define STAT_EX_ERASE  0x6
#define STAT_EX_CREATE 0x7

#define ERRMEMD_SEQ_NR_TWICE        0x08000000
#define ERRMEMD_START               0x10000000
#define ERRMEMD_MULTIPLE_ALLOC_FAIL 0x20000000
#define ERRMEMD_MESSAGE_MISSING     0x40000000
#define ERRMEMD_SEQ_NR_MISSMATCH    0x80000000

/* If no alignement with which messages shall be
 * stored in the persistent memory is specified,
 * this default alignment will be taken for storages 
 * of type BLOCK. For all other devices the
 * specification of an alignment is mandatory.
 */
#define DEFAULT_ALIGNMENT  0x4

#define BACKEND_COPY       0x1
#define BACKEND_MOVE       0x2

/* This magic is hashed with the currently valid 
 * block magic in order to identify a message as
 * valid in the current life cycle of the 
 * persistent storage. The variable magic from
 * struct PersistentMessage stores this hashed value.
 */

#define MAGIC_MESSAGE 0x96c3f1e7

/* Vm message Macro */
#ifndef OWNER_NONE
#define OWNER_NONE	0
#define OWNER_VMM	1
#endif

#define VMM_STRING	"VMM>"
#define VM_STRING	"VM[%hu]>"
#define VM_STRING_SZ (strlen(VM_STRING))
#define VM_STRING_SZ_MAX (VM_STRING_SZ + ERRMEM_MAX_ENTRY_LENGTH)

#ifndef ERRMEM_GET_VM
#define ERRMEM_GET_VM(flags) ((flags >> 16) & 0xff)
#define ERRMEM_SET_VM(flags, vm) (flags |= ((vm & 0xff) << 16))
#define ERRMEM_MASK_VM(flags) (flags & 0xff00ffff)
#define ERRMEM_USER_RESERVED 0xff000000
#endif

struct ErrmemBackend;
struct msg_header;

typedef struct ErrmemBackendDev {
	uint32_t type;           /* backend type */
	uint32_t access_options; /* checks for create / backup */
	uint32_t nr;
	uint32_t size;
	uint32_t cnt;
	uint32_t persistent;
	uint32_t w_width;
	uint32_t wait;
	uint32_t retry;
	int32_t  debug;
	char*    name;
	char*    command;
} ErrmemBackendDev_t;

typedef struct PersistentMessage_old {
	uint32_t magic;
	uint32_t blocknum;
	uint32_t len;
	struct errmem_message *msg;
} PersistentMessage_old_t;

typedef struct PersistentMessage {
	uint32_t magic;
	uint32_t blocknum;
	uint16_t crc;
	uint16_t reserved;
	uint32_t len;
	struct errmem_message *msg;
} PersistentMessage_t;

typedef struct MessageQueue {	
	struct errmem_message  msg;
	struct MessageQueue*   next;
} MessageQueue_t;

typedef struct SessionInfo {
	int32_t   fd;         /* current file handele */
	uint32_t  block_cur;  /* flag to indicate current block as currently used */
	uint32_t  pos;        /* file handle position */
	uint32_t  num;        /* real block number (0..block count) */
	uint32_t  num_start;  /* real block number where reading starts */
	uint32_t  seq_num;    /* seq number of current block of file */
	uint32_t  seq_start;  /* seq number of block or file where reading starts */
	uint32_t  seq_max;    /* highest uses sequence number of block of file */
	uint32_t  block_magic;/* currently valid block magic knwon by session */ 
	int32_t   status;     /* session status (possible error code) */
	int32_t   state;      /* session state as result of status */
	uint32_t  seq_last;   /* seq number of last read block or file */
	uint32_t  seq_next;   /* seq number of next block or file to read */
	uint32_t  seq_new;    /* indicates the beginning of a new block */
	uint32_t  client_mode;/* indicates whether read is done by daemon or app */ 
	uint32_t  persistent; /* number of existing persistent blocks or files */
	void*     block;      /* pointer to block data in case of block device */
	char*     fn;         /* name of current file referenced by fd */
	struct ErrmemBackend* backend;  /* assigned backend */
	struct MessageQueue* msg_queue; /* msg from storage incl. generated msgs */
} SessionInfo_t;

// ErrmemBackend - generic interface for an error memory backend
//
// An error memory backend can store or otherwise output error memory
// messages. Different backends are created by filling below struct
// with different function pointers.
//
// Error memory backends are organized in a forward-linked list,
// allowing to call a function for all elements of the list.
typedef struct ErrmemBackend {
	// get the command to execute for this device
	char* (*get_command)(struct ErrmemBackend*);
	// get the path name of the device
	char* (*get_name)(struct ErrmemBackend*);
	// get position of the device in commandline
	unsigned (*get_nr)(struct ErrmemBackend*);
	// set position of backend in case a backend cannot be created
	void (*set_nr)(struct ErrmemBackend*, uint32_t);
	// get default flag of backend
	int32_t (*is_default)(struct ErrmemBackend*);
	// set default flag of  a backend
	void (*set_default)(struct ErrmemBackend*, int32_t);
	// get type of backend
	unsigned (*get_type)(struct ErrmemBackend*);
	// get sequence number from last stored message, before daemon was stopped
	unsigned (*get_last_seq_nr)(struct ErrmemBackend*);
	// info - print information about the backend on fd
	void (*info) (struct ErrmemBackend*, int fd);
	// dump - dump the contents of the backend on another backend
	void (*dump) (struct ErrmemBackend*, struct ErrmemBackend*,
				  struct SessionInfo*, const char*, int, int);
	// erase - erase the contents of the backend
	signed (*erase) (struct ErrmemBackend*);
	// store - store an error memory entry in the backend
	unsigned (*store) (struct ErrmemBackend*,
			   int len, struct errmem_message*);
	// create - create the error memory backend
    int32_t (*create) (int32_t, struct ErrmemBackend*);
	// destroy - destroy the error memory backend
	void (*destroy) (struct ErrmemBackend*);
	// open a context for a client to access this backend
	void (*open_session)(struct SessionInfo* si);
	// read a message from this backend
	void (*read_session)(struct SessionInfo* si);
	// pointer to next backend
	struct ErrmemBackend* next;
} ErrmemBackend_t;

/*
 * Implementation of an error memory backend that dumps error memory
 * messages into a buffer. Defined here because it is additionally
 * used by the file backend and tcpip backend (may be future use).
 */
typedef struct ErrmemBackendBuffered {
	ErrmemBackend_t backend; // error memory backend interface
	unsigned size;           // size of buffer to dump messages
	unsigned type;
	char* buffer;            // buffer to dump messages
} ErrmemBackendBuffered_t;

static inline char* int_stat_type2char_stat_type(int32_t val)
{
	if (val      == 0140000)
		return "S_IFSOCK";
	else if (val == 0120000)
		return "S_IFLNK";
	else if (val == 0100000)
		return "S_IFREG";
	else if (val ==  060000)
		return "S_IFBLK";
	else if (val ==  040000)
		return "S_IFDIR";
	else if (val ==  020000)
		return "S_IFCHR";
	else if (val ==  010000)
		return "S_IFIFO";
	else
		return "UNKNOWN_STAT_TYPE";
}

static inline char* int_type2char_type(int32_t val)
{
	if (val == 0x1)
		return "FILE_FS";
	else if (val == 0x2)
		return "BLOCK";
	else if (val == 0x4)
		return "BLOCK_RAW";
	else if (val == 0x8)
		return "BLOCK_FLASH";
	else if (val == 0x10)
		return "BLOCK_FLASH_RAW";
	else if (val == 0x20)
		return "BLOCK_NOR";
	else if (val == 0x40)
		return "BLOCK_NOR_RAW";
	else if (val == 0x100)
		return "BLOCK_NAND";
	else if (val == 0x200)
		return "BLOCK_NAND_RAW";
	else if (val == 0x400)
		return "BUFFER";
	else if (val == 0x10000)
		return "DLT";
	else
		return "UNKNOWN_DEVICE_TYPE";
}

// addptr - add an offset to a pointer (and avoid a lint warning)
static inline void* addptr(unsigned char* m, uint32_t o)
{
	return m + o;
}

uint16_t calc_crc16(uint16_t seed, const uint8_t *verify , size_t len);

/* open the device name with mode access_mode */
int32_t open_dev(const char* name, int access_mode, int suppress);

/* read from device referenced by fd size bytes of data into buf */
/* name is used for error messages in case of need */
int32_t read_dev(int32_t fd, const char* name, char* buf, size_t size);

/* seek to position in device */
int32_t seek_dev(int32_t fd, char* name, off_t pos, int32_t mode);

/* close the persitent storage */
int32_t close_dev(int fd, char* name);

/* write into the persistent storage */
int32_t write_dev(int32_t fd, char* name, char* buf, size_t size);

void * get_mem(uint32_t size);

// errmem_backend_info
// - call info() on a list of error memory backends
void errmem_backend_info(ErrmemBackend_t* backend, int fd);

// errmem_backend_erase_content
// - call erase() on a list of error memory backends
void errmem_backend_erase_content(
	ErrmemBackend_t* backend);

// errmem_backend_store
// - call store() on a list of error memory backends
unsigned errmem_backend_store(
	ErrmemBackend_t** queue, ErrmemBackend_t** rm,
	int len, struct errmem_message* msg);

// errmem_backend_create
// - creates the backend having given access to the underlying storage
void errmem_backend_create(
	int32_t  access,
	ErrmemBackend_t** pbackend);

// errmem_backend_destroy
// - call destroy() on a list of error memory backends
void errmem_backend_destroy(ErrmemBackend_t* backend);

// determines the requested backend type (cmd line param) and init creation
ErrmemBackend_t* errmem_backend_factory(ErrmemBackendDev_t* bdev);

// creates data structure for a block device storage
ErrmemBackend_t* create_block_structure(ErrmemBackendDev_t* p_dev);

// creates data structure for a file device storage
ErrmemBackend_t* create_file_structure(ErrmemBackendDev_t* p_dev);

// creates interface structure to write data to DLT interface
ErrmemBackend_t* create_dlt_structure(ErrmemBackendDev_t* p_dev);

// create backend to stream data to a requested output channel
ErrmemBackend_t* create_channel_structure(ErrmemBackendDev_t* p_dev);

ErrmemBackend_t* errmem_backend_create_buffered(ErrmemBackendDev_t*);

ErrmemBackend_t* errmem_backend_create_tcpip(
	ErrmemBackend_t*, int fd);

MessageQueue_t** errmem_get_message(struct msg_header* p, MessageQueue_t** mq);

void errmem_add_to_mqueue(MessageQueue_t* queue, MessageQueue_t* msg);

void errmem_create_messages(SessionInfo_t *si);

int32_t errmem_add_backend(ErrmemBackend_t** queue, ErrmemBackend_t* b);

int32_t errmem_remove_backend(ErrmemBackend_t** queue, ErrmemBackend_t* b);

#endif // ifndef _ERRMEM_BACKEND_H
